<?php

namespace GoDaddy\WordPress\MWC\Core\Features\EmailNotifications\Traits;

use DOMDocument;
use Exception;
use GoDaddy\WordPress\MWC\Common\Helpers\ArrayHelper;
use GoDaddy\WordPress\MWC\Common\Helpers\TypeHelper;
use GoDaddy\WordPress\MWC\Common\Register\Register;
use GoDaddy\WordPress\MWC\Common\Register\Types\RegisterFilter;
use GoDaddy\WordPress\MWC\Core\Features\EmailNotifications\Contracts\EmailTemplateContract;
use Pelago\Emogrifier;
use Pelago\Emogrifier\CssInliner;
use Pelago\Emogrifier\HtmlProcessor\CssToAttributeConverter;
use Pelago\Emogrifier\HtmlProcessor\HtmlPruner;
use RuntimeException;
use Throwable;
use WC_Email;

/**
 * A trait for classes that generate from WooCommerce email objects.
 */
trait CanGetWooCommerceEmailOutputTrait
{
    /** @var string[] */
    protected $wooCommerceTemplateOptions = [
        'woocommerce_email_background_color'      => 'container.backgroundColor',
        'woocommerce_email_base_color'            => 'header.backgroundColor',
        'woocommerce_email_body_background_color' => 'body.backgroundColor',
        'woocommerce_email_footer_text'           => 'footer.footerText',
        'woocommerce_email_header_image'          => 'header.image.url',
        'woocommerce_email_text_color'            => 'body.text.color',
    ];

    /** @var RegisterFilter[] */
    protected $wooCommerceTemplateOptionFilters = [];

    /** @var array<mixed> configuration from a {@see EmailTemplateContract} instance */
    protected $emailTemplateConfiguration = [];

    /** @var string|null CSS rules applied to WooCommerce email content */
    protected $wooCommerceEmailStyles;

    /**
     * Sets the configuration of the email template.
     *
     * @param array<mixed> $value a configuration array likely generated with {@see EmailTemplateContract::getConfiguration()}
     * @return void
     */
    protected function setEmailTemplateConfiguration(array $value) : void
    {
        $this->emailTemplateConfiguration = $value;
    }

    /**
     * Gets the configuration of the email template.
     *
     * @return array<mixed>
     */
    protected function getEmailTemplateConfiguration() : array
    {
        return $this->emailTemplateConfiguration;
    }

    /**
     * Sets the email template configuration using the information from the given template instance.
     *
     * @param EmailTemplateContract|null $emailTemplate optional email template instance
     * @return void
     */
    protected function setConfigurationFromEmailTemplate(?EmailTemplateContract $emailTemplate = null) : void
    {
        if ($emailTemplate) {
            $this->setEmailTemplateConfiguration($emailTemplate->getConfiguration());
        }
    }

    /**
     * Adds inline style attributes for the HTML code generated by WooCommerce email hooks using WooCommerce's CSS for emails.
     *
     * @see WC_Email::style_inline()
     *
     * @param string $content
     * @return string
     */
    protected function addInlineStyles(string $content) : string
    {
        if (! $content) {
            return $content;
        }

        $css = $this->getWooCommerceEmailStyles();

        if ($newContent = $this->getContentWithInlineStyles($content, $css)) {
            return $newContent;
        }

        return '<style type="text/css">'.$css.'</style>'.$content;
    }

    /**
     * Gets WooCommerce CSS rules for emails.
     *
     * TODO: add tests for this method {wvega 2021-10-13}
     *
     * @return string
     */
    protected function getWooCommerceEmailStyles() : string
    {
        if (is_null($this->wooCommerceEmailStyles)) {
            $this->wooCommerceEmailStyles = $this->loadWooCommerceEmailStyles();
        }

        return $this->wooCommerceEmailStyles;
    }

    /**
     * Gets WooCommerce CSS rules for emails.
     *
     * TODO: add tests for this method {wvega 2021-10-13}
     *
     * @return string
     */
    protected function loadWooCommerceEmailStyles() : string
    {
        $styles = $this->getOutputFromCallback(function () {
            wc_get_template('emails/email-styles.php');
        });

        if (! $wooCommerceEmail = $this->getWooCommerceEmail()) {
            return $styles;
        }

        return (string) apply_filters('woocommerce_email_styles', $styles, $wooCommerceEmail);
    }

    /**
     * Gets the WooCommerce email object.
     *
     * @return WC_Email|null
     */
    abstract protected function getWooCommerceEmail();

    /**
     * Gets the given HTML content after adding the provided CSS styles as inline attributes.
     *
     * @param string $content HTML content
     * @param string $css CSS styles
     * @return string|null
     */
    protected function getContentWithInlineStyles(string $content, string $css) : ?string
    {
        // WooCommerce 6.5+
        if ($newContent = $this->getContentWithInlineStylesUsingCssInliner($content, $css)) {
            return $newContent;
        }

        // WooCommerce 6.4.* or older
        return $this->getContentWithInlineStylesUsingEmogrifier($content, $css);
    }

    /**
     * Gets the given HTML content after adding the provided CSS styles as inline attributes using the {@see CssInliner} class.
     *
     * The {@see CssInliner} class is available when WooCommerce 6.5+ is installed.
     *
     * @link https://developer.woocommerce.com/2022/04/04/breaking-change-notice-woocommerce_emogrifier-hook/
     *
     * @param string $content HTML content
     * @param string $css CSS styles
     * @return string|null
     */
    protected function getContentWithInlineStylesUsingCssInliner(string $content, string $css) : ?string
    {
        try {
            return $this->getBodyContentFromDocument($this->getDomDocumentWithInlineStylesUsingCssInliner($content, $css));
        } catch (Throwable $throwable) {
            return null;
        }
    }

    /**
     * Uses {@see CssInliner} to apply the given inline CSS to the given HTML content.
     *
     * @param string $content
     * @param string $css
     * @return DOMDocument
     * @throws Exception
     */
    public function getDomDocumentWithInlineStylesUsingCssInliner(string $content, string $css) : DOMDocument
    {
        $cssInliner = $this->getCssInlinerFromHtml($content)->inlineCss($css);

        $this->maybeDoWooCommerceEmogrifierAction($cssInliner);

        return $cssInliner->getDomDocument();
    }

    /**
     * Attempts to run the woocommerce_emogrifier action using the given object and an {@see WC_Email} instance, if available.
     *
     * @param object $object
     */
    protected function maybeDoWooCommerceEmogrifierAction(object $object) : void
    {
        if (! $email = $this->getWooCommerceEmail()) {
            return;
        }

        do_action('woocommerce_emogrifier', $object, $email);
    }

    /**
     * Gets an instance of {@see CssInliner} for the given HTML content.
     *
     * @param string $html
     * @return CssInliner
     * @throws RuntimeException
     */
    protected function getCssInlinerFromHtml(string $html)
    {
        if (! class_exists(CssInliner::class)) {
            throw new RuntimeException('Class CssInliner does not exist.');
        }

        return CssInliner::fromHtml($html);
    }

    /**
     * Gets HTML for the body element of the given HTML document.
     *
     * @param DOMDocument $document
     * @return string
     * @throws RuntimeException
     */
    protected function getBodyContentFromDocument(DOMDocument $document) : string
    {
        return $this->getCssToAttributeConverterFromDomDocument($this->removeElementsWithDisplayNone($document))
            ->convertCssToVisualAttributes()
            ->renderBodyContent();
    }

    /**
     * Removes elements that have display: none from the given document.
     *
     * @param DOMDocument $document
     * @return DOMDocument
     * @throws RuntimeException
     */
    protected function removeElementsWithDisplayNone(DOMDocument $document) : DOMDocument
    {
        $this->getHtmlPrunnerFromDomDocument($document)->removeElementsWithDisplayNone();

        return $document;
    }

    /**
     * Creates a new instance of {@see HtmlPruner} to process the given document.
     *
     * @param DOMDocument $document
     * @return HtmlPruner
     * @throws RuntimeException
     */
    protected function getHtmlPrunnerFromDomDocument(DOMDocument $document)
    {
        if (! class_exists(HtmlPruner::class)) {
            throw new RuntimeException('Class HtmlPrunner does not exist.');
        }

        return HtmlPruner::fromDomDocument($document);
    }

    /**
     * Creates a new instance of {@see CssToAttributeConverter} to process the given document.
     *
     * @param DOMDocument $document
     * @return CssToAttributeConverter
     * @throws RuntimeException
     */
    protected function getCssToAttributeConverterFromDomDocument(DOMDocument $document)
    {
        if (! class_exists(CssToAttributeConverter::class)) {
            throw new RuntimeException('Class CssToAttributeConverter does not exist.');
        }

        return CssToAttributeConverter::fromDomDocument($document);
    }

    /**
     * Gets the given HTML content after adding the provided CSS styles as inline attributes using the {@see Emogrifier} class.
     *
     * The {@see Emogrifier} class is available when WooCommerce 6.4.* or older is installed.
     *
     * @param string $content HTML content
     * @param string $css CSS styles
     * @return string|null
     */
    protected function getContentWithInlineStylesUsingEmogrifier(string $content, string $css) : ?string
    {
        try {
            return $this->getBodyContentFromHtml($this->getHtmlWithInlineStylesUsingEmogrifier($content, $css));
        } catch (Throwable $exception) {
            return null;
        }
    }

    /**
     * Uses {@see Emogrifier} to apply the given CSS to the given HTML content.
     *
     * @param string $content HTML content
     * @param string $css inline CSS to apply
     * @return string
     * @throws RuntimeException
     */
    protected function getHtmlWithInlineStylesUsingEmogrifier(string $content, string $css) : string
    {
        $emogrifier = $this->getEmogrifier($content, $css);

        $this->maybeDoWooCommerceEmogrifierAction($emogrifier);

        return $emogrifier->emogrify();
    }

    /**
     * Creates a new instance of {@see Emogrifier} using the given HTML and CSS.
     *
     * @param string $html
     * @param string $css
     * @return Emogrifier
     * @throws RuntimeException
     */
    protected function getEmogrifier(string $html, string $css)
    {
        if (! class_exists(Emogrifier::class)) {
            throw new RuntimeException('Class Emogrifier does not exist.');
        }

        return new Emogrifier($html, $css);
    }

    /**
     * Gets the HTML for the body element of the given HTML content.
     *
     * @param string $html
     * @return string
     * @throws RuntimeException
     */
    protected function getBodyContentFromHtml(string $html) : string
    {
        return $this->getHtmlPrunnerFromHtml($html)->removeElementsWithDisplayNone()->renderBodyContent();
    }

    /**
     * Creates a new instance of {@see HtmlPruner} to process the given HTML content.
     *
     * @param string $html
     * @return HtmlPruner
     * @throws RuntimeException
     */
    protected function getHtmlPrunnerFromHtml(string $html)
    {
        if (! class_exists(HtmlPruner::class)) {
            throw new RuntimeException('Class HtmlPrunner does not exist.');
        }

        return HtmlPruner::fromHtml($html);
    }

    /**
     * Executes the given callback and returns its output.
     *
     * @param callable $callback
     * @return string
     */
    protected function getOutputFromCallback(callable $callback) : string
    {
        return (string) $this->tryOutputBufferingCallback(function () use ($callback) {
            ob_start();

            $callback();

            return (string) ob_get_clean();
        });
    }

    /**
     * Executes the given callback and catches all exceptions and errors thrown.
     *
     * It closes any output buffers that remain opened when an exception or error occurs.
     *
     * @param callable $callback a callable that returns content
     * @return string|null
     */
    protected function tryOutputBufferingCallback(callable $callback)
    {
        $outputBufferingLevel = ob_get_level();

        try {
            return $callback();
        } catch (Throwable $exception) {
            // ignore all Exceptions and Errors
        } finally {
            // make sure that all output buffers are closed
            // this finally block will be executed even if the try block returns
            while (ob_get_level() > $outputBufferingLevel) {
                ob_end_clean();
            }
        }

        return null;
    }

    /**
     * Registers filters to override the value of several WooCommerce email template options.
     *
     * @return void
     */
    protected function temporarilyOverrideWooCommerceTemplateOptions() : void
    {
        foreach (array_keys($this->wooCommerceTemplateOptions) as $optionName) {
            /** @var RegisterFilter $filter */
            $filter = Register::filter()
                ->setGroup("option_{$optionName}")
                ->setHandler([$this, 'overrideWooCommerceTemplateOption'])
                ->setArgumentsCount(2);

            try {
                $filter->execute();
            } catch (Exception $exception) {
                // move on if there is an error trying to register the filter
                continue;
            }

            $this->wooCommerceTemplateOptionFilters[] = $filter;
        }
    }

    /**
     * Filters the value of the given option if it is one of the WooCommerce email template options with overrides.
     *
     * @param mixed $value the current value of the option
     * @param string|mixed $option the name of the option that we are filtering
     * @return array<mixed>|mixed
     */
    public function overrideWooCommerceTemplateOption($value, $option)
    {
        if (! $setting = TypeHelper::string(ArrayHelper::get($this->wooCommerceTemplateOptions, TypeHelper::string($option, '')), '')) {
            return $value;
        }

        return ArrayHelper::get($this->getEmailTemplateConfiguration(), $setting, $value);
    }

    /**
     * De-registers the filters used to override the value WooCommerce email template options.
     *
     * @return void
     * @throws Exception
     */
    protected function restoreWooCommerceTemplateOptions() : void
    {
        foreach ($this->wooCommerceTemplateOptionFilters as $filter) {
            $filter->deregister();
        }

        $this->wooCommerceTemplateOptionFilters = [];
    }
}
